import React, {
  useEffect,
  useMemo,
  useState,
  useCallback,
  useRef,
} from "react";
import confetti from "canvas-confetti";

/*
  Single-file React app (App.jsx)
  - Dashboard with levels
  - Game with match-3 logic
  - 5-second confetti on level complete -> auto next level
  - Back button stops confetti immediately
  - LocalStorage save for unlocked level & best scores
*/

// ----- Level definitions -----
const LEVELS = [
  { id: 1, target: 500, moves: 20, boardSize: 8 },
  { id: 2, target: 1200, moves: 20, boardSize: 8 },
  { id: 3, target: 1500, moves: 20, boardSize: 9 },
  { id: 4, target: 1700, moves: 20, boardSize: 9 },
];

const CANDIES = ["🔴", "🟠", "🟡", "🟢", "🔵", "🟣"];
const randCandy = () => CANDIES[Math.floor(Math.random() * CANDIES.length)];

// ---- PREVENT INITIAL MATCHES ----
function initializeGrid(size) {
  const g = Array.from({ length: size }, () =>
    Array.from({ length: size }, () => randCandy())
  );

  let changed = true;
  while (changed) {
    changed = false;
    for (let r = 0; r < size; r++) {
      for (let c = 0; c < size; c++) {
        const cur = g[r][c];
        if (c >= 2 && g[r][c - 1] === cur && g[r][c - 2] === cur) {
          g[r][c] = randCandy();
          changed = true;
        }
        if (r >= 2 && g[r - 1][c] === cur && g[r - 2][c] === cur) {
          g[r][c] = randCandy();
          changed = true;
        }
      }
    }
  }
  return g;
}

function findMatches(grid) {
  const size = grid.length;
  const set = new Set();
  const add = (r, c) => set.add(`${r},${c}`);

  // horizontal
  for (let r = 0; r < size; r++) {
    for (let c = 0; c < size - 2; c++) {
      const a = grid[r][c];
      if (!a) continue;
      if (grid[r][c + 1] === a && grid[r][c + 2] === a) {
        add(r, c);
        add(r, c + 1);
        add(r, c + 2);
        let k = c + 3;
        while (k < size && grid[r][k] === a) {
          add(r, k);
          k++;
        }
      }
    }
  }

  // vertical
  for (let c = 0; c < size; c++) {
    for (let r = 0; r < size - 2; r++) {
      const a = grid[r][c];
      if (!a) continue;
      if (grid[r + 1][c] === a && grid[r + 2][c] === a) {
        add(r, c);
        add(r + 1, c);
        add(r + 2, c);
        let k = r + 3;
        while (k < size && grid[k][c] === a) {
          add(k, c);
          k++;
        }
      }
    }
  }

  return Array.from(set).map((s) => s.split(",").map(Number));
}

function applyGravityAndRefill(grid) {
  const size = grid.length;
  const g = grid.map((r) => [...r]);

  for (let c = 0; c < size; c++) {
    let write = size - 1;
    for (let r = size - 1; r >= 0; r--) {
      if (g[r][c] !== null) {
        g[write][c] = g[r][c];
        write--;
      }
    }
    for (let r = write; r >= 0; r--) g[r][c] = randCandy();
  }
  return g;
}

// ----- LOCAL STORAGE -----
const LS_UNLOCK = "candy_unlocked_level";
const LS_BEST = "candy_best_scores";

// ============================================================
//                      MAIN APP
// ============================================================
export default function App() {
  const [view, setView] = useState("dashboard");
  const [selectedLevel, setSelectedLevel] = useState(null);
  const [unlocked, setUnlocked] = useState(
    Number(localStorage.getItem(LS_UNLOCK)) || 1
  );
  const [bestScores, setBestScores] = useState(
    JSON.parse(localStorage.getItem(LS_BEST) || "{}")
  );

  useEffect(() => {
    localStorage.setItem(LS_UNLOCK, String(unlocked));
  }, [unlocked]);

  useEffect(() => {
    localStorage.setItem(LS_BEST, JSON.stringify(bestScores));
  }, [bestScores]);

  const startLevel = (lvl) => {
    setSelectedLevel(lvl);
    setView("game");
  };

  // update best/unlock (but do not navigate)
  const onLevelComplete = (lvl, score) => {
    setBestScores((p) => ({
      ...p,
      [lvl]: Math.max(p[lvl] || 0, score),
    }));
    if (lvl < LEVELS.length && unlocked <= lvl) setUnlocked(lvl + 1);
  };

  // Move to next level in-place (no dashboard)
  const onNextLevel = (nextLevel) => {
    if (nextLevel) {
      setSelectedLevel(nextLevel);
      setView("game");
    } else {
      // no next level -> go dashboard
      setView("dashboard");
    }
  };

  return (
    <div style={styles.app}>
      <header style={styles.header}>
        <h1 style={{ margin: 0 }}>Sweet Swap — Dashboard</h1>
      </header>

      {view === "dashboard" && (
        <div style={styles.dashboardWrap}>
          <div style={styles.levelGrid}>
            {LEVELS.map((lvl) => (
              <div key={lvl.id} style={styles.card}>
                <div style={styles.cardHeader}>
                  <strong>Level {lvl.id}</strong>
                  <span style={{ fontSize: 12, color: "#bbb" }}>
                    Target: {lvl.target}
                  </span>
                </div>

                <div style={{ marginTop: 8 }}>
                  <div style={{ fontSize: 13, color: "#ddd" }}>
                    Moves: {lvl.moves} • Board: {lvl.boardSize}×{lvl.boardSize}
                  </div>
                </div>

                <div style={{ marginTop: 12, display: "flex", gap: 8 }}>
                  {lvl.id <= unlocked ? (
                    <button
                      style={styles.playBtn}
                      onClick={() => startLevel(lvl)}
                    >
                      ▶ Play
                    </button>
                  ) : (
                    <button style={styles.lockBtn} disabled>
                      🔒 Locked
                    </button>
                  )}

                  <div style={styles.bestBox}>
                    Best: {bestScores[lvl.id] || 0}
                  </div>
                </div>
              </div>
            ))}
          </div>

          <div style={styles.footerRow}>
            <div>Unlocked Level: {unlocked}</div>
            <div>
              <button
                style={styles.smallBtn}
                onClick={() => {
                  localStorage.clear();
                  setUnlocked(1);
                  setBestScores({});
                }}
              >
                Reset Progress
              </button>
            </div>
          </div>
        </div>
      )}

      {view === "game" && selectedLevel && (
        <Game
          level={selectedLevel}
          onBack={() => setView("dashboard")}
          onComplete={onLevelComplete}
          onNextLevel={onNextLevel}
        />
      )}
    </div>
  );
}

// ============================================================
//                      GAME COMPONENT
// ============================================================
function Game({ level, onBack, onComplete, onNextLevel }) {
  const size = level.boardSize;
  const confettiRef = useRef(null);
  const finishingRef = useRef(false); // prevent double-trigger

  const [grid, setGrid] = useState(() => initializeGrid(size));
  const [selected, setSelected] = useState(null);
  const [score, setScore] = useState(0);
  const [movesLeft, setMovesLeft] = useState(level.moves);
  const [processing, setProcessing] = useState(false);
  const [message, setMessage] = useState("");

  // Reset internal state when `level` changes (moving to next level)
  useEffect(() => {
    setGrid(initializeGrid(level.boardSize));
    setSelected(null);
    setScore(0);
    setMovesLeft(level.moves);
    setProcessing(false);
    setMessage("");
    finishingRef.current = false;
    stopConfetti();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [level.id]);

  // Confetti controller: start and stop (5 seconds)
  const startConfettiFiveSeconds = (callback) => {
    stopConfetti(); // clear previous if any
    const duration = 5000;
    const end = Date.now() + duration;

    confettiRef.current = setInterval(() => {
      confetti({
        particleCount: 70,
        spread: 90,
        startVelocity: 40,
        origin: { x: Math.random(), y: -0.1 }, // falling from top
      });

      if (Date.now() >= end) {
        stopConfetti();
        callback && callback();
      }
    }, 250);
  };

  const stopConfetti = () => {
    if (confettiRef.current) {
      clearInterval(confettiRef.current);
      confettiRef.current = null;
    }
  };

  // when score reaches target -> start confetti and move to next level after 5s
  useEffect(() => {
    if (score >= level.target && !finishingRef.current) {
      finishingRef.current = true;
      setMessage("🎉 Level Complete!");

      // Start confetti for 5 seconds; after it ends -> update best/unlock + next
      startConfettiFiveSeconds(() => {
        onComplete(level.id, score);

        // find next level
        const nextId = level.id + 1;
        const nextLevel = LEVELS.find((l) => l.id === nextId);

        if (nextLevel) {
          // move to next level (in-place)
          onNextLevel(nextLevel);
        } else {
          // last level completed → go back to dashboard
          onNextLevel(null);
        }
      });
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [score]);

  // process matches loop
  const processAllMatches = useCallback(async (g) => {
    setProcessing(true);
    let temp = g.map((r) => [...r]);
    let gained = 0;

    while (true) {
      const matches = findMatches(temp);
      if (matches.length === 0) break;

      gained += matches.length * 10;

      matches.forEach(([r, c]) => (temp[r][c] = null));
      setGrid(temp.map((r) => [...r]));
      // small visual pause
      // eslint-disable-next-line no-await-in-loop
      await new Promise((res) => setTimeout(res, 220));

      temp = applyGravityAndRefill(temp);
      setGrid(temp.map((r) => [...r]));
      // eslint-disable-next-line no-await-in-loop
      await new Promise((res) => setTimeout(res, 160));
    }

    setScore((s) => s + gained);
    setProcessing(false);
  }, []);

  const handleCellClick = (r, c) => {
    if (processing) return;
    if (movesLeft <= 0) return;

    if (!selected) return setSelected([r, c]);

    const [sr, sc] = selected;
    const adj = Math.abs(sr - r) + Math.abs(sc - c) === 1;
    if (!adj) return setSelected([r, c]);

    const g = grid.map((row) => [...row]);
    [g[sr][sc], g[r][c]] = [g[r][c], g[sr][sc]];
    setGrid(g);
    setSelected(null);
    setMovesLeft((m) => m - 1);

    const matches = findMatches(g);
    if (matches.length) {
      processAllMatches(g);
    } else {
      setTimeout(() => {
        setGrid((prev) => {
          const back = prev.map((row) => [...row]);
          [back[sr][sc], back[r][c]] = [back[r][c], back[sr][sc]];
          return back;
        });
      }, 220);

      setMessage("No match — try again");
      setTimeout(() => setMessage(""), 900);
    }
  };

  // GAME OVER
  useEffect(() => {
    if (movesLeft <= 0 && score < level.target) {
      setMessage("Game Over — Out of moves");
      setTimeout(() => {
        stopConfetti();
        onBack();
      }, 1400);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [movesLeft, score]);

  const resetLevel = () => {
    setGrid(initializeGrid(size));
    setScore(0);
    setMovesLeft(level.moves);
    setSelected(null);
    setMessage("");
    finishingRef.current = false;
    stopConfetti();
  };

  // Back button handler: stop confetti immediately & return dashboard
  const handleBack = () => {
    stopConfetti();
    finishingRef.current = false;
    onBack();
  };

  return (
    <div style={styles.gameWrap}>
      <div style={styles.topBar}>
        <button style={styles.smallBtn} onClick={handleBack}>
          ← Back
        </button>

        <div>
          <div style={{ fontWeight: 700 }}>Level {level.id}</div>
          <div style={{ fontSize: 13, color: "#ccc" }}>
            Target: {level.target}
          </div>
        </div>

        <div style={{ display: "flex", gap: 10 }}>
          <div style={styles.statBox}>Score: {score}</div>
          <div style={styles.statBox}>Moves: {movesLeft}</div>
        </div>
      </div>

      {message && <div style={styles.message}>{message}</div>}

      <div style={{ marginTop: 12 }}>
        <div
          style={{
            display: "grid",
            gap: 6,
            gridTemplateColumns: `repeat(${size}, 46px)`,
          }}
        >
          {grid.map((row, r) =>
            row.map((cell, c) => (
              <div
                key={`${r}-${c}`}
                onClick={() => handleCellClick(r, c)}
                style={{
                  width: 46,
                  height: 46,
                  background:
                    selected && selected[0] === r && selected[1] === c
                      ? "#ffd6e0"
                      : "#1f2937",
                  display: "flex",
                  alignItems: "center",
                  justifyContent: "center",
                  fontSize: 22,
                  borderRadius: 6,
                  cursor: processing ? "wait" : "pointer",
                }}
              >
                {cell}
              </div>
            ))
          )}
        </div>
      </div>

      <div style={{ marginTop: 14, display: "flex", gap: 8 }}>
        <button style={styles.primaryBtn} onClick={resetLevel}>
          Restart
        </button>
      </div>
    </div>
  );
}

// ============================================================
//                      STYLES
// ============================================================
const styles = {
  app: {
    minHeight: "100vh",
    background: "linear-gradient(180deg,#08121a,#0b1220)",
    color: "#e6eef8",
    padding: 18,
    fontFamily:
      "Inter, ui-sans-serif, system-ui, -apple-system, 'Segoe UI', Roboto, 'Helvetica Neue', Arial",
  },
  header: {
    marginBottom: 18,
  },
  dashboardWrap: {
    maxWidth: 920,
    margin: "0 auto",
  },
  levelGrid: {
    display: "grid",
    gridTemplateColumns: "repeat(auto-fit, minmax(220px, 1fr))",
    gap: 12,
  },
  card: {
    background: "linear-gradient(180deg,#0f1724,#071022)",
    padding: 12,
    borderRadius: 10,
    border: "1px solid rgba(255,255,255,0.04)",
    boxShadow: "0 6px 18px rgba(2,6,23,0.6)",
  },
  cardHeader: {
    display: "flex",
    justifyContent: "space-between",
  },
  playBtn: {
    background: "#ff6b9a",
    border: "none",
    color: "white",
    padding: "8px 12px",
    borderRadius: 8,
    cursor: "pointer",
  },
  lockBtn: {
    background: "#334155",
    border: "none",
    color: "#94a3b8",
    padding: "8px 12px",
    borderRadius: 8,
  },
  bestBox: {
    marginLeft: 8,
    padding: "6px 8px",
    background: "#0b1220",
    borderRadius: 8,
    color: "#ffd9b3",
    fontWeight: 700,
  },
  footerRow: {
    marginTop: 18,
    display: "flex",
    justifyContent: "space-between",
  },
  smallBtn: {
    background: "#273449",
    border: "1px solid rgba(255,255,255,0.04)",
    color: "#dbeafe",
    padding: "6px 10px",
    borderRadius: 8,
    cursor: "pointer",
  },
  gameWrap: { maxWidth: 520, margin: "0 auto", textAlign: "center" },
  topBar: {
    display: "flex",
    justifyContent: "space-between",
    marginBottom: 8,
  },
  statBox: {
    background: "#071022",
    padding: "6px 10px",
    borderRadius: 8,
    minWidth: 84,
  },
  message: {
    marginTop: 8,
    padding: 10,
    background: "#064e3b",
    color: "#bbf7d0",
    borderRadius: 8,
  },
  primaryBtn: {
    background: "#ff6b9a",
    border: "none",
    color: "white",
    padding: "8px 14px",
    borderRadius: 8,
    cursor: "pointer",
  },
  secondaryBtn: {
    background: "#334155",
    border: "none",
    color: "#cbd5e1",
    padding: "8px 14px",
    borderRadius: 8,
    cursor: "pointer",
  },
};
